import pool from '../config/db.js';
import { getSimpleMarket } from './coingecko.js';
let ioRef = null;
const feedsByVs = {};
let defaultIntervalMs = 30000;
let defaultVsSet = new Set();
async function fetchActiveCoins() {
    const [rows] = await pool.query(`SELECT id, symbol, coingecko_id as coingeckoId
		 FROM coins
		 WHERE is_active = 1 AND coingecko_id IS NOT NULL AND coingecko_id <> ''`);
    return (rows || []).map(r => ({ id: Number(r.id), symbol: String(r.symbol).toUpperCase(), coingeckoId: String(r.coingeckoId).toLowerCase() }));
}
async function tickForVs(vsCurrency) {
    if (!ioRef)
        return;
    const vs = vsCurrency.toUpperCase();
    try {
        const coins = await fetchActiveCoins();
        const ids = Array.from(new Set(coins.map(c => c.coingeckoId).filter(Boolean)));
        if (ids.length === 0)
            return;
        const market = await getSimpleMarket(ids, vs, { ttlMs: 60000 });
        const items = coins.map(c => {
            const m = market[c.coingeckoId];
            return {
                coinId: c.id,
                coingeckoId: c.coingeckoId,
                symbol: c.symbol,
                price: m?.price ?? null,
                change24h: typeof m?.change24h === 'number' ? m.change24h : undefined,
                vol24h: typeof m?.vol24h === 'number' ? m.vol24h : undefined,
            };
        }).filter(i => typeof i.price === 'number' && isFinite(i.price));
        const snapshot = { vsCurrency: vs, timestamp: Date.now(), items };
        feedsByVs[vs].lastSnapshot = snapshot;
        ioRef.to(`market:${vs}`).emit('rates:update', snapshot);
    }
    catch (err) {
        // swallow errors; will try again on next tick
    }
}
function startTimerForVs(vsCurrency) {
    const vs = vsCurrency.toUpperCase();
    if (!feedsByVs[vs])
        feedsByVs[vs] = { refCount: 0 };
    const feed = feedsByVs[vs];
    if (feed.interval)
        return; // already running
    // Immediate tick then interval
    tickForVs(vs);
    feed.interval = setInterval(() => tickForVs(vs), defaultIntervalMs);
}
function stopTimerForVs(vsCurrency) {
    const vs = vsCurrency.toUpperCase();
    const feed = feedsByVs[vs];
    if (!feed)
        return;
    if (feed.interval) {
        clearInterval(feed.interval);
        feed.interval = undefined;
    }
}
export function startMarketFeed(io, options) {
    ioRef = io;
    defaultIntervalMs = Math.max(5000, Number(options?.intervalMs ?? 30000));
    defaultVsSet = new Set((options?.defaultVs || ['USD']).map(v => v.toUpperCase()));
    for (const vs of defaultVsSet) {
        startTimerForVs(vs);
    }
}
export function ensureVsFeed(vsCurrency) {
    const vs = vsCurrency.toUpperCase();
    if (!feedsByVs[vs])
        feedsByVs[vs] = { refCount: 0 };
    feedsByVs[vs].refCount += 1;
    startTimerForVs(vs);
}
export function releaseVsFeed(vsCurrency) {
    const vs = vsCurrency.toUpperCase();
    const feed = feedsByVs[vs];
    if (!feed)
        return;
    feed.refCount = Math.max(0, (feed.refCount || 0) - 1);
    // Keep default VS feeds always running; stop others when no subscribers
    if (feed.refCount === 0 && !defaultVsSet.has(vs)) {
        stopTimerForVs(vs);
    }
}
export function getLatestSnapshot(vsCurrency) {
    const vs = vsCurrency.toUpperCase();
    return feedsByVs[vs]?.lastSnapshot;
}
