import fs from 'fs';
import path from 'path';
import { fileURLToPath } from 'url';
import Handlebars from 'handlebars';
import { getEmailConfig } from '../../config/email.js';
import { NodemailerProvider } from './providers/NodemailerProvider.js';
import { MailgunProvider } from './providers/MailgunProvider.js';
const __dirname = path.dirname(fileURLToPath(import.meta.url));
const templatesDir = path.join(__dirname, '..', '..', '..', 'assets', 'emails');
function ensureTemplateExists(name, ext) {
    const fp = path.join(templatesDir, `${name}.${ext}`);
    return fs.existsSync(fp) ? fp : null;
}
class TemplateRenderer {
    constructor() {
        this.cache = new Map();
    }
    renderHtml(name, vars) {
        const file = ensureTemplateExists(name, 'hbs');
        if (!file)
            return undefined;
        let compiled = this.cache.get(file);
        if (!compiled) {
            const src = fs.readFileSync(file, 'utf8');
            compiled = Handlebars.compile(src);
            this.cache.set(file, compiled);
        }
        return compiled(vars);
    }
    renderText(name, vars) {
        const file = ensureTemplateExists(name, 'txt');
        if (!file)
            return undefined;
        let compiled = this.cache.get(file);
        if (!compiled) {
            const src = fs.readFileSync(file, 'utf8');
            compiled = Handlebars.compile(src);
            this.cache.set(file, compiled);
        }
        return compiled(vars);
    }
}
export class EmailService {
    constructor() {
        this.primary = null;
        this.fallback = null;
        this.renderer = new TemplateRenderer();
        const cfg = getEmailConfig();
        this.from = cfg.from;
        this.disabled = cfg.disableSend;
        if (cfg.provider === 'nodemailer' && cfg.smtp) {
            this.primary = new NodemailerProvider(cfg.smtp, this.from);
        }
        else if (cfg.provider === 'mailgun' && cfg.mailgun) {
            this.primary = new MailgunProvider(cfg.mailgun, this.from);
        }
        if (cfg.fallbackProvider) {
            if (cfg.fallbackProvider === 'nodemailer' && cfg.smtp) {
                this.fallback = new NodemailerProvider(cfg.smtp, this.from);
            }
            else if (cfg.fallbackProvider === 'mailgun' && cfg.mailgun) {
                this.fallback = new MailgunProvider(cfg.mailgun, this.from);
            }
        }
    }
    get isDisabled() { return this.disabled; }
    async send(template, to, subject, variables) {
        const html = this.renderer.renderHtml(template, variables);
        const text = this.renderer.renderText(template, variables);
        if (this.disabled) {
            try {
                console.warn('[EmailService] Sending disabled; template=%s to=%s', template, to);
            }
            catch { }
            return { id: undefined, provider: 'disabled' };
        }
        const payload = { to, subject, html, text, from: this.from };
        try {
            if (!this.primary)
                throw new Error('No primary email provider configured');
            const res = await this.primary.send(payload);
            return { id: res.id, provider: 'primary' };
        }
        catch (e) {
            try {
                console.error('[EmailService] Primary provider failed:', e);
            }
            catch { }
            if (this.fallback) {
                try {
                    const res = await this.fallback.send(payload);
                    return { id: res.id, provider: 'fallback' };
                }
                catch (e2) {
                    try {
                        console.error('[EmailService] Fallback provider failed:', e2);
                    }
                    catch { }
                    throw e2;
                }
            }
            throw e;
        }
    }
    async sendPasswordReset(to, token, vars) {
        const cfg = getEmailConfig();
        const resetUrl = `${cfg.appBaseUrl.replace(/\/$/, '')}/reset-password?token=${encodeURIComponent(token)}`;
        const variables = {
            userName: vars?.userName || 'there',
            resetUrl,
            expiresMinutes: vars?.expiresMinutes ?? 30,
            appName: process.env.APP_NAME || 'Crypto Wallet',
        };
        return this.send('password-reset', to, 'Reset your password', variables);
    }
    async sendVerifyEmail(to, token, vars) {
        const cfg = getEmailConfig();
        const verifyUrl = `${cfg.appBaseUrl.replace(/\/$/, '')}/verify-email?token=${encodeURIComponent(token)}`;
        const variables = {
            userName: vars?.userName || 'there',
            verifyUrl,
            expiresMinutes: vars?.expiresMinutes ?? 60,
            appName: process.env.APP_NAME || 'Crypto Wallet',
        };
        return this.send('verify-email', to, 'Verify your email', variables);
    }
    async sendTwoStepCode(to, code, vars) {
        const variables = {
            userName: vars?.userName || 'there',
            code,
            expiresMinutes: vars?.expiresMinutes ?? 10,
            appName: process.env.APP_NAME || 'Crypto Wallet',
        };
        return this.send('two-step', to, 'Your verification code', variables);
    }
    async sendWithdrawalRequested(to, payload) {
        const variables = {
            userName: payload.userName || 'there',
            amount: payload.amount,
            asset: payload.asset,
            network: payload.network || 'N/A',
            address: payload.address,
            requestId: String(payload.requestId),
            appName: process.env.APP_NAME || 'Crypto Wallet',
        };
        return this.send('withdrawal', to, 'Withdrawal requested', variables);
    }
    async sendDepositConfirmed(to, payload) {
        const variables = {
            userName: payload.userName || 'there',
            amount: payload.amount,
            asset: payload.asset,
            network: payload.network || 'N/A',
            txHash: payload.txHash || 'N/A',
            appName: process.env.APP_NAME || 'Crypto Wallet',
        };
        return this.send('deposit', to, 'Deposit received', variables);
    }
}
let singleton = null;
export function getEmailService() {
    if (!singleton)
        singleton = new EmailService();
    return singleton;
}
