import https from 'https';
const cacheByVs = {};
function httpsGetJson(url, timeoutMs = 8000) {
    return new Promise((resolve, reject) => {
        const req = https.get(url, { timeout: timeoutMs, headers: { 'User-Agent': 'crypto-wallet-server/1.0' } }, (res) => {
            const status = res.statusCode || 0;
            if (status < 200 || status >= 300) {
                res.resume();
                reject(new Error(`HTTP ${status} fetching ${url}`));
                return;
            }
            let data = '';
            res.setEncoding('utf8');
            res.on('data', (chunk) => { data += chunk; });
            res.on('end', () => {
                try {
                    const json = JSON.parse(data);
                    resolve(json);
                }
                catch (err) {
                    reject(err);
                }
            });
        });
        req.on('timeout', () => {
            req.destroy(new Error('Request timed out'));
        });
        req.on('error', (err) => reject(err));
    });
}
/**
 * Fetch prices from CoinGecko Simple Price API for a set of coin ids and a vs currency.
 * - Minimizes calls by using an in-memory TTL cache per coin id and currency
 * - Batches all requested ids into a single request when any are stale/missing
 */
export async function getSimplePrices(coinGeckoIds, vsCurrency, options) {
    const ids = Array.from(new Set((coinGeckoIds || []).map((s) => String(s).toLowerCase()).filter(Boolean)));
    const vs = String(vsCurrency || 'USD').toLowerCase();
    const ttlMs = Math.max(1000, Number(options?.ttlMs ?? 60000));
    if (!cacheByVs[vs])
        cacheByVs[vs] = {};
    const cache = cacheByVs[vs];
    const now = Date.now();
    const missingOrStale = [];
    for (const id of ids) {
        const entry = cache[id];
        if (!entry || now - entry.fetchedAtMs > ttlMs)
            missingOrStale.push(id);
    }
    if (missingOrStale.length > 0) {
        const url = `https://api.coingecko.com/api/v3/simple/price?ids=${encodeURIComponent(missingOrStale.join(','))}&vs_currencies=${encodeURIComponent(vs)}`;
        try {
            const resp = await httpsGetJson(url);
            for (const [id, obj] of Object.entries(resp || {})) {
                const price = Number(obj?.[vs]);
                if (isFinite(price)) {
                    const prev = cache[id] || { fetchedAtMs: now };
                    cache[id] = { ...prev, price, fetchedAtMs: now };
                }
            }
        }
        catch {
            // On failure, do not throw; return what we have in cache (may be empty)
        }
    }
    const result = {};
    for (const id of ids) {
        const price = cache[id]?.price;
        if (typeof price === 'number' && isFinite(price))
            result[id] = price;
    }
    return result;
}
export async function getSimpleMarket(coinGeckoIds, vsCurrency, options) {
    const ids = Array.from(new Set((coinGeckoIds || []).map((s) => String(s).toLowerCase()).filter(Boolean)));
    const vs = String(vsCurrency || 'USD').toLowerCase();
    const ttlMs = Math.max(1000, Number(options?.ttlMs ?? 60000));
    if (!cacheByVs[vs])
        cacheByVs[vs] = {};
    const cache = cacheByVs[vs];
    const now = Date.now();
    const missingOrStale = [];
    for (const id of ids) {
        const entry = cache[id];
        if (!entry || now - entry.fetchedAtMs > ttlMs)
            missingOrStale.push(id);
    }
    if (missingOrStale.length > 0) {
        const url = `https://api.coingecko.com/api/v3/simple/price?ids=${encodeURIComponent(missingOrStale.join(','))}&vs_currencies=${encodeURIComponent(vs)}&include_24hr_change=true&include_24hr_vol=true`;
        try {
            const resp = await httpsGetJson(url);
            for (const [id, obj] of Object.entries(resp || {})) {
                const price = Number(obj?.[vs]);
                const change = Number(obj?.[`${vs}_24h_change`] ?? obj?.[`${vs}_24hr_change`] ?? obj?.[`${vs}_24h_change_percent`]);
                const vol = Number(obj?.[`${vs}_24h_vol`] ?? obj?.[`${vs}_24hr_vol`] ?? obj?.[`${vs}_24h_volume`]);
                if (isFinite(price)) {
                    cache[id] = { price, change24h: isFinite(change) ? change : undefined, vol24h: isFinite(vol) ? vol : undefined, fetchedAtMs: now };
                }
            }
        }
        catch {
            // ignore network errors, keep cache
        }
    }
    const result = {};
    for (const id of ids) {
        const entry = cache[id];
        if (entry && isFinite(entry.price))
            result[id] = { price: entry.price, change24h: entry.change24h, vol24h: entry.vol24h };
    }
    return result;
}
export async function getSimpleVolume24h(coinGeckoIds, vsCurrency, options) {
    const ids = Array.from(new Set((coinGeckoIds || []).map((s) => String(s).toLowerCase()).filter(Boolean)));
    const vs = String(vsCurrency || 'USD').toLowerCase();
    const ttlMs = Math.max(1000, Number(options?.ttlMs ?? 60000));
    if (!cacheByVs[vs])
        cacheByVs[vs] = {};
    const cache = cacheByVs[vs];
    const now = Date.now();
    const missingOrStale = [];
    for (const id of ids) {
        const entry = cache[id];
        if (!entry || now - entry.fetchedAtMs > ttlMs)
            missingOrStale.push(id);
    }
    if (missingOrStale.length > 0) {
        const url = `https://api.coingecko.com/api/v3/simple/price?ids=${encodeURIComponent(missingOrStale.join(','))}&vs_currencies=${encodeURIComponent(vs)}`;
        try {
            const resp = await httpsGetJson(url);
            for (const [id, obj] of Object.entries(resp || {})) {
                const volume = Number(obj?.[`${vs}_24h_vol`]);
                if (isFinite(volume)) {
                    cache[id] = { price: volume, fetchedAtMs: now };
                }
            }
        }
        catch {
            // On failure, do not throw; return what we have in cache (may be empty)
        }
    }
    const result = {};
    for (const id of ids) {
        const volume = cache[id]?.price;
        if (typeof volume === 'number' && isFinite(volume))
            result[id] = volume;
    }
    return result;
}
