import { Router } from 'express';
import { z } from 'zod';
import pool from '../config/db.js';
const router = Router();
const depositSchema = z.object({
    userId: z.string(),
    walletId: z.string(),
    coinId: z.number(),
    amount: z.string(), // string decimal
    txHash: z.string().min(3),
    fromAddress: z.string().optional(),
    toAddress: z.string().optional(),
    network: z.string().optional(),
    confirmations: z.number().optional(),
});
router.post('/deposit', async (req, res) => {
    const parsed = depositSchema.safeParse(req.body);
    if (!parsed.success)
        return res.status(400).json({ error: 'Invalid payload', issues: parsed.error.issues });
    const { userId, walletId, coinId, amount, txHash, fromAddress, toAddress, network, confirmations } = parsed.data;
    const conn = await pool.getConnection();
    try {
        // Ensure wallet belongs to user
        const [own] = await conn.query('SELECT id FROM user_wallets WHERE id = :walletId AND user_id = :userId LIMIT 1', { walletId, userId });
        if (!own || own.length === 0)
            return res.status(404).json({ error: 'Wallet not found' });
        // Idempotency: check tx hash exists
        const [exists] = await conn.query('SELECT id FROM transactions WHERE tx_hash = :txHash LIMIT 1', { txHash });
        if (exists && exists.length > 0)
            return res.status(200).json({ ok: true, duplicate: true });
        await conn.query(`INSERT INTO transactions 
       (public_id, user_id, user_wallet_id, coin_id, tx_type, source, status, amount, fee, from_address, to_address, tx_hash, network, confirmations)
       VALUES (CONCAT(REPLACE(UUID(), '-', ''), SUBSTRING(REPLACE(UUID(), '-', ''), 1, 20)), :userId, :walletId, :coinId, 'deposit', 'chain', 'completed', :amount, 0, :fromAddress, :toAddress, :txHash, :network, :confirmations)`, { userId, walletId, coinId, amount, fromAddress, toAddress, txHash, network, confirmations: confirmations ?? 0 });
        try {
            const [emailRow] = await conn.query(`SELECT email, first_name FROM users WHERE id = :id LIMIT 1`, { id: userId });
            const [coinRow] = await conn.query(`SELECT symbol FROM coins WHERE id = :id LIMIT 1`, { id: coinId });
            if (emailRow && emailRow.length) {
                const email = String(emailRow[0].email);
                const firstName = String(emailRow[0].first_name || 'there');
                const asset = coinRow && coinRow.length ? String(coinRow[0].symbol) : '';
                const { getEmailService } = await import('../services/email/index.js');
                await getEmailService().sendDepositConfirmed(email, { userName: firstName, amount, asset, network: network || null, txHash });
            }
        }
        catch { }
        return res.json({ ok: true });
    }
    finally {
        conn.release();
    }
});
export default router;
