import { Router } from 'express';
import { z } from 'zod';
import pool from '../config/db.js';
import { authGuard } from '../middleware/auth.js';
import { roleGuard } from '../middleware/roles.js';
import path from 'path';
import fs from 'fs';
import { fileURLToPath } from 'url';
const router = Router();
const __dirname = path.dirname(fileURLToPath(import.meta.url));
const assetsDir = path.join(__dirname, '..', '..', 'assets', 'coins');
if (!fs.existsSync(assetsDir))
    fs.mkdirSync(assetsDir, { recursive: true });
router.get('/settings', authGuard, roleGuard([1]), async (_req, res) => {
    const [rows] = await pool.query(`SELECT * FROM system_settings WHERE id = 1`);
    res.json(rows[0] || null);
});
const updateSchema = z.object({
    app_name: z.string().min(1),
    domain: z.string().optional().nullable(),
    smtp_host: z.string().optional().nullable(),
    smtp_port: z.coerce.number().optional().nullable(),
    smtp_user: z.string().optional().nullable(),
    smtp_pass: z.string().optional().nullable(),
    smtp_secure: z.coerce.boolean().optional(),
    from_email: z.string().optional().nullable(),
    support_display_name: z.preprocess((v) => {
        if (typeof v === 'string' && v.trim() === '')
            return undefined;
        return v;
    }, z.string().min(1).optional()),
});
router.post('/settings', authGuard, roleGuard([1]), async (req, res) => {
    const parsed = updateSchema.safeParse(req.body);
    if (!parsed.success)
        return res.status(400).json({ error: 'Invalid input' });
    const s = parsed.data;
    await pool.query(`UPDATE system_settings SET app_name = :app_name, domain = :domain, smtp_host = :smtp_host, smtp_port = :smtp_port, smtp_user = :smtp_user, smtp_pass = :smtp_pass, smtp_secure = :smtp_secure, from_email = :from_email, support_display_name = COALESCE(:support_display_name, support_display_name) WHERE id = 1`, s);
    const [rows] = await pool.query(`SELECT * FROM system_settings WHERE id = 1`);
    res.json(rows[0]);
});
router.get('/coins', authGuard, roleGuard([1]), async (_req, res) => {
    const [rows] = await pool.query(`SELECT id, symbol, name, network, decimals, is_active, coingecko_id, logo_url FROM coins WHERE is_active = 1 ORDER BY id ASC`);
    res.json(rows);
});
const addCoinSchema = z.object({ symbol: z.string().min(2), name: z.string().min(2), coingecko_id: z.string().min(2), logo_url: z.string().url().optional().nullable(), network: z.string().optional().nullable(), decimals: z.number().int().min(0).max(36).default(18) });
async function downloadToFile(url, dest) {
    const res = await fetch(url);
    if (!res.ok || !res.body)
        throw new Error('Failed to download');
    const reader = res.body.getReader();
    const chunks = [];
    while (true) {
        const { done, value } = await reader.read();
        if (done)
            break;
        if (value)
            chunks.push(value);
    }
    const buffer = Buffer.concat(chunks.map((u) => Buffer.from(u)));
    await fs.promises.writeFile(dest, buffer);
}
router.post('/coins', authGuard, roleGuard([1]), async (req, res) => {
    const parsed = addCoinSchema.safeParse(req.body);
    if (!parsed.success)
        return res.status(400).json({ error: 'Invalid input' });
    const c = parsed.data;
    // Prepare local logo if provided
    let localUrl = null;
    try {
        if (c.logo_url) {
            const filename = `${c.symbol.toUpperCase()}.png`;
            const savePath = path.join(assetsDir, filename);
            await downloadToFile(c.logo_url, savePath);
            localUrl = `/assets/coins/${filename}`;
        }
    }
    catch {
        localUrl = null;
    }
    // Check existing coin by symbol
    const [existing] = await pool.query(`SELECT id, is_active FROM coins WHERE symbol = :symbol LIMIT 1`, { symbol: c.symbol });
    if (existing && existing.length) {
        const ex = existing[0];
        if (ex.is_active === 1) {
            return res.status(409).json({ error: 'Coin exists' });
        }
        // Reactivate and update metadata
        await pool.query(`UPDATE coins SET name = :name, network = :network, decimals = :decimals, is_active = 1, coingecko_id = :coingecko_id, logo_url = :logo_url WHERE id = :id`, { ...c, logo_url: localUrl, id: ex.id });
        return res.json({ id: ex.id, logo_url: localUrl });
    }
    // Insert new coin
    const [r] = await pool.query(`INSERT INTO coins (symbol, name, network, decimals, is_active, coingecko_id, logo_url)
     VALUES (:symbol, :name, :network, :decimals, 1, :coingecko_id, :logo_url)`, { ...c, logo_url: localUrl });
    res.status(201).json({ id: r.insertId, logo_url: localUrl });
});
router.delete('/coins/:id', authGuard, roleGuard([1]), async (req, res) => {
    const { id } = req.params;
    await pool.query(`UPDATE coins SET is_active = 0 WHERE id = :id`, { id });
    res.json({ ok: true });
});
export default router;
