import { Router } from 'express';
import { z } from 'zod';
import pool from '../config/db.js';
import { authGuard } from '../middleware/auth.js';
import { roleGuard } from '../middleware/roles.js';
import multer from 'multer';
import path from 'path';
import fs from 'fs';
import { fileURLToPath } from 'url';
import { getIO, emitToUser, isUserOnline } from '../socket.js';
const router = Router();
const upload = multer({ limits: { fileSize: 10 * 1024 * 1024 } });
const __dirname = path.dirname(fileURLToPath(import.meta.url));
const supportAssetsBase = path.join(__dirname, '..', '..', 'assets', 'support');
if (!fs.existsSync(supportAssetsBase))
    fs.mkdirSync(supportAssetsBase, { recursive: true });
async function resolveCurrentUserId(uid) {
    const [rows] = await pool.query(`SELECT id, role_id FROM users WHERE uid = :uid LIMIT 1`, { uid });
    if (!rows || !rows.length)
        return null;
    return { id: Number(rows[0].id), roleId: Number(rows[0].role_id) };
}
async function resolveUserIdByUid(uid) {
    const [rows] = await pool.query(`SELECT id FROM users WHERE uid = :uid LIMIT 1`, { uid });
    return rows && rows.length ? Number(rows[0].id) : null;
}
// Helper to enforce agent ownership
async function ensureAgentOwnsTarget(agentUid, targetUserId) {
    const [me] = await pool.query(`SELECT id FROM users WHERE uid = :uid LIMIT 1`, { uid: agentUid });
    if (!me || !me.length)
        return false;
    const myId = Number(me[0].id);
    const [own] = await pool.query(`SELECT COUNT(*) as cnt FROM users WHERE id = :id AND owner_id = :owner LIMIT 1`, { id: targetUserId, owner: myId });
    return own && own.length && Number(own[0].cnt) > 0;
}
// Create ticket (client or staff on behalf of a client)
const createTicketSchema = z.object({ subject: z.string().min(1), userUid: z.string().optional() });
router.post('/tickets', authGuard, async (req, res) => {
    const actor = req.user;
    const parsed = createTicketSchema.safeParse(req.body);
    if (!parsed.success)
        return res.status(400).json({ error: 'Invalid input' });
    const actorRec = await resolveCurrentUserId(actor.sub);
    if (!actorRec)
        return res.status(401).json({ error: 'Unauthorized' });
    let targetUserId = actorRec.id;
    if (actor.role !== 3 && parsed.data.userUid) {
        const uidUserId = await resolveUserIdByUid(parsed.data.userUid);
        if (!uidUserId)
            return res.status(404).json({ error: 'User not found' });
        // Agent must own the user
        if (actor.role === 2) {
            const owns = await ensureAgentOwnsTarget(actor.sub, uidUserId);
            if (!owns)
                return res.status(403).json({ error: 'Forbidden' });
        }
        targetUserId = uidUserId;
    }
    const [r] = await pool.query(`INSERT INTO support_tickets (public_id, user_id, subject) VALUES (REPLACE(UUID(), '-', ''), :userId, :subject)`, { userId: targetUserId, subject: parsed.data.subject });
    // Emit new-ticket to admins and owner agent (if exists)
    try {
        const [ownerRow] = await pool.query(`SELECT u.owner_id as ownerId, (SELECT uid FROM users WHERE id = u.owner_id) as ownerUid FROM users u WHERE u.id = :id LIMIT 1`, { id: targetUserId });
        const ownerUid = ownerRow && ownerRow.length ? (ownerRow[0].ownerUid ? String(ownerRow[0].ownerUid) : null) : null;
        if (ownerUid)
            emitToUser(ownerUid, 'support:new-ticket', { ticketId: r.insertId, userId: targetUserId });
        getIO().to('role:admin').emit('support:new-ticket', { ticketId: r.insertId, userId: targetUserId });
    }
    catch { }
    return res.status(201).json({ id: r.insertId });
});
// List tickets for current user (client), agent-owned users, or all (admin)
router.get('/tickets', authGuard, async (req, res) => {
    const actor = req.user;
    const q = String(req.query.q || '');
    const status = String(req.query.status || '');
    const limit = Math.min(Number(req.query.limit || 50), 200);
    const offset = Math.max(Number(req.query.offset || 0), 0);
    const actorRec = await resolveCurrentUserId(actor.sub);
    if (!actorRec)
        return res.status(401).json({ error: 'Unauthorized' });
    const params = { limit, offset };
    const whereClauses = [];
    if (q) {
        whereClauses.push('(t.subject LIKE :q OR u.email LIKE :q OR u.first_name LIKE :q OR u.last_name LIKE :q)');
        params.q = `%${q}%`;
    }
    if (status && ['open', 'closed'].includes(status)) {
        whereClauses.push('t.status = :status');
        params.status = status;
    }
    if (actor.role === 3) {
        params.userId = actorRec.id;
        whereClauses.push('t.user_id = :userId');
    }
    else if (actor.role === 2) {
        // Restrict to users owned by the agent
        params.owner = actorRec.id;
        whereClauses.push('t.user_id IN (SELECT id FROM users WHERE owner_id = :owner)');
    }
    const where = whereClauses.length ? whereClauses.join(' AND ') : '1=1';
    const [rows] = await pool.query(`SELECT t.id, t.public_id as publicId, t.user_id as userId, t.subject, t.status, t.created_at as createdAt, t.updated_at as updatedAt,
		        u.email as userEmail, u.first_name as userFirstName, u.last_name as userLastName, u.uid as userUid,
		        (
		          SELECT COUNT(*) FROM support_messages m
		          WHERE m.ticket_id = t.id
		            AND m.created_at > COALESCE((SELECT r.last_read_at FROM support_ticket_reads r WHERE r.ticket_id = t.id AND r.user_id = :viewerId LIMIT 1), '1970-01-01')
		            AND (m.sender_user_id IS NULL OR m.sender_user_id <> :viewerId)
		        ) AS unreadCount
		 FROM support_tickets t
		 JOIN users u ON u.id = t.user_id
		 WHERE ${where}
		 ORDER BY t.updated_at DESC
		 LIMIT :limit OFFSET :offset`, { ...params, viewerId: actorRec.id });
    res.json({ items: rows, limit, offset });
});
// Close or open a ticket (admin or owning agent). Admin can also rename display name of staff label via system settings route separately.
const updateTicketSchema = z.object({ status: z.enum(['open', 'closed']) });
router.patch('/tickets/:id', authGuard, roleGuard([1, 2]), async (req, res) => {
    const actor = req.user;
    const id = Number(req.params.id);
    const parsed = updateTicketSchema.safeParse(req.body);
    if (!parsed.success)
        return res.status(400).json({ error: 'Invalid input' });
    // Ownership for agent
    if (actor.role === 2) {
        const [row] = await pool.query(`SELECT user_id FROM support_tickets WHERE id = :id LIMIT 1`, { id });
        if (!row || !row.length)
            return res.status(404).json({ error: 'Not found' });
        const userId = Number(row[0].user_id);
        const owns = await ensureAgentOwnsTarget(actor.sub, userId);
        if (!owns)
            return res.status(403).json({ error: 'Forbidden' });
    }
    await pool.query(`UPDATE support_tickets SET status = :status WHERE id = :id`, { status: parsed.data.status, id });
    return res.json({ ok: true });
});
// Get ticket messages
router.get('/tickets/:id/messages', authGuard, async (req, res) => {
    const actor = req.user;
    const id = Number(req.params.id);
    // Check access
    const [tick] = await pool.query(`SELECT user_id FROM support_tickets WHERE id = :id LIMIT 1`, { id });
    if (!tick || !tick.length)
        return res.status(404).json({ error: 'Not found' });
    const ticketUserId = Number(tick[0].user_id);
    if (actor.role === 3) {
        const me = await resolveCurrentUserId(actor.sub);
        if (!me || me.id !== ticketUserId)
            return res.status(403).json({ error: 'Forbidden' });
    }
    if (actor.role === 2) {
        const owns = await ensureAgentOwnsTarget(actor.sub, ticketUserId);
        if (!owns)
            return res.status(403).json({ error: 'Forbidden' });
    }
    const [rows] = await pool.query(`SELECT m.id, m.sender_role as senderRole, m.sender_user_id as senderUserId, m.message, m.created_at as createdAt
		 FROM support_messages m WHERE m.ticket_id = :id ORDER BY m.created_at ASC`, { id });
    // Load attachments in bulk
    const ids = rows.map(r => Number(r.id));
    let attachmentsByMsg = {};
    if (ids.length) {
        const [atts] = await pool.query(`SELECT message_id as messageId, url, filename FROM support_attachments WHERE message_id IN (${ids.map(() => '?').join(',')})`, ids);
        for (const a of (atts || [])) {
            const mid = Number(a.messageId);
            if (!attachmentsByMsg[mid])
                attachmentsByMsg[mid] = [];
            attachmentsByMsg[mid].push({ url: a.url, filename: a.filename });
        }
    }
    const items = rows.map(r => ({ ...r, attachments: attachmentsByMsg[Number(r.id)] || [] }));
    // Upsert read marker for this viewer
    try {
        const [viewer] = await pool.query(`SELECT id FROM users WHERE uid = :uid LIMIT 1`, { uid: actor.sub });
        const viewerId = viewer && viewer.length ? Number(viewer[0].id) : null;
        if (viewerId) {
            await pool.query(`INSERT INTO support_ticket_reads (ticket_id, user_id, last_read_at)
				 VALUES (:ticketId, :userId, NOW())
				 ON DUPLICATE KEY UPDATE last_read_at = NOW()`, { ticketId: id, userId: viewerId });
        }
    }
    catch { }
    res.json({ items });
});
// Post message (text only)
const postMessageSchema = z.object({ message: z.string().optional() });
router.post('/tickets/:id/messages', authGuard, async (req, res) => {
    const actor = req.user;
    const id = Number(req.params.id);
    const parsed = postMessageSchema.safeParse(req.body);
    if (!parsed.success)
        return res.status(400).json({ error: 'Invalid input' });
    // Access check (same as GET) + ensure ticket open
    const [tick] = await pool.query(`SELECT user_id, status FROM support_tickets WHERE id = :id LIMIT 1`, { id });
    if (!tick || !tick.length)
        return res.status(404).json({ error: 'Not found' });
    const ticketUserId = Number(tick[0].user_id);
    const ticketStatus = String(tick[0].status);
    if (ticketStatus !== 'open')
        return res.status(400).json({ error: 'Ticket is closed' });
    if (actor.role === 3) {
        const me = await resolveCurrentUserId(actor.sub);
        if (!me || me.id !== ticketUserId)
            return res.status(403).json({ error: 'Forbidden' });
    }
    if (actor.role === 2) {
        const owns = await ensureAgentOwnsTarget(actor.sub, ticketUserId);
        if (!owns)
            return res.status(403).json({ error: 'Forbidden' });
    }
    // Insert message
    const me = await resolveCurrentUserId(actor.sub);
    const senderRole = actor.role === 3 ? 'client' : 'staff';
    const [r] = await pool.query(`INSERT INTO support_messages (ticket_id, sender_user_id, sender_role, message) VALUES (:ticketId, :senderUserId, :senderRole, :message)`, { ticketId: id, senderUserId: me?.id ?? null, senderRole, message: parsed.data.message ?? null });
    // Figure recipients
    const [uids] = await pool.query(`SELECT u.uid as userUid, u.owner_id as ownerId, (SELECT uid FROM users WHERE id = u.owner_id) as ownerUid FROM users u WHERE u.id = :id LIMIT 1`, { id: ticketUserId });
    const userUid = uids && uids.length ? String(uids[0].userUid) : null;
    const ownerUid = uids && uids.length && uids[0].ownerUid ? String(uids[0].ownerUid) : null;
    const ownerId = uids && uids.length && uids[0].ownerId ? Number(uids[0].ownerId) : null;
    // Live updates
    try {
        if (userUid)
            emitToUser(userUid, 'support:new-message', { ticketId: id });
        if (ownerUid)
            emitToUser(ownerUid, 'support:new-message', { ticketId: id });
        getIO().to('role:admin').emit('support:new-message', { ticketId: id });
    }
    catch { }
    // Offline notifications (DB) – only when recipient offline
    async function insertNotification(userId, payload) {
        await pool.query(`INSERT INTO notifications (user_id, type, title, message, level) VALUES (:userId, 'support', :title, :message, 'info')`, { userId, title: payload.title, message: payload.message || null });
    }
    try {
        if (senderRole === 'staff') {
            // notify client if offline (check by uid)
            if (userUid && !isUserOnline(userUid)) {
                await insertNotification(ticketUserId, { title: 'New Support Message', message: (parsed.data.message || '').slice(0, 120) });
            }
        }
        else {
            // client sent -> notify owner agent if offline (check by uid)
            if (ownerId && ownerUid && !isUserOnline(ownerUid)) {
                await insertNotification(ownerId, { title: 'New Support Message', message: (parsed.data.message || '').slice(0, 120) });
            }
        }
    }
    catch { }
    return res.status(201).json({ id: r.insertId });
});
// Upload attachment for a message
router.post('/tickets/:id/messages/:messageId/attachments', authGuard, upload.single('file'), async (req, res) => {
    const actor = req.user;
    const id = Number(req.params.id);
    const messageId = Number(req.params.messageId);
    if (!req.file)
        return res.status(400).json({ error: 'File required' });
    // Validate access to ticket
    const [tick] = await pool.query(`SELECT user_id FROM support_tickets WHERE id = :id LIMIT 1`, { id });
    if (!tick || !tick.length)
        return res.status(404).json({ error: 'Not found' });
    const ticketUserId = Number(tick[0].user_id);
    if (actor.role === 3) {
        const me = await resolveCurrentUserId(actor.sub);
        if (!me || me.id !== ticketUserId)
            return res.status(403).json({ error: 'Forbidden' });
    }
    if (actor.role === 2) {
        const owns = await ensureAgentOwnsTarget(actor.sub, ticketUserId);
        if (!owns)
            return res.status(403).json({ error: 'Forbidden' });
    }
    // Save file under assets/support/<userUid>/ticket-<ticketId>/
    const [u] = await pool.query(`SELECT uid FROM users WHERE id = :id LIMIT 1`, { id: ticketUserId });
    const uid = u && u.length ? String(u[0].uid) : 'unknown';
    const dir = path.join(supportAssetsBase, uid, `ticket-${id}`);
    fs.mkdirSync(dir, { recursive: true });
    const safeName = `${Date.now()}-${req.file.originalname.replace(/[^a-zA-Z0-9._-]/g, '_')}`;
    const savePath = path.join(dir, safeName);
    fs.writeFileSync(savePath, req.file.buffer);
    const relUrl = `/assets/support/${uid}/ticket-${id}/${safeName}`;
    await pool.query(`INSERT INTO support_attachments (message_id, url, filename, mime_type, size) VALUES (:messageId, :url, :filename, :mime, :size)`, { messageId, url: relUrl, filename: req.file.originalname, mime: req.file.mimetype, size: req.file.size });
    return res.status(201).json({ url: relUrl });
});
// Admin-only: delete a ticket (cascade deletes messages/attachments)
router.delete('/tickets/:id', authGuard, roleGuard([1]), async (req, res) => {
    const id = Number(req.params.id);
    await pool.query(`DELETE FROM support_tickets WHERE id = :id`, { id });
    return res.json({ ok: true });
});
export default router;
