import { Router } from 'express';
import { authGuard } from '../middleware/auth.js';
import pool from '../config/db.js';
import { getLatestSnapshot } from '../services/marketFeed.js';
const router = Router();
router.get('/coins', async (_req, res) => {
    const [rows] = await pool.query(`SELECT id, symbol, name, coingecko_id, logo_url FROM coins WHERE is_active = 1 ORDER BY id ASC`);
    res.json(rows);
});
router.get('/app-info', async (_req, res) => {
    const [rows] = await pool.query(`SELECT app_name, support_display_name FROM system_settings WHERE id = 1`);
    const row = rows && rows.length ? rows[0] : { app_name: 'CryptoWallet', support_display_name: 'Support' };
    res.json({ appName: row.app_name, supportDisplayName: row.support_display_name || 'Support' });
});
// List my active deposit routes (client)
router.get('/me/deposit-routes', authGuard, async (req, res) => {
    const user = req.user;
    if (!user)
        return res.status(401).json({ error: 'Unauthorized' });
    const [u] = await pool.query(`SELECT id FROM users WHERE uid = :uid LIMIT 1`, { uid: user.sub });
    if (!u || !u.length)
        return res.status(404).json({ error: 'Not found' });
    const userId = Number(u[0].id);
    const [rows] = await pool.query(`SELECT id, kind, scope, currency, bank_name as bankName, account_holder as accountHolder, account_number as accountNumber, iban, swift, routing, instructions
		 FROM deposit_routes WHERE user_id = :uid AND is_active = 1 ORDER BY created_at DESC`, { uid: userId });
    return res.json({ items: rows });
});
// Get current market snapshot
router.get('/market-snapshot', async (req, res) => {
    try {
        const vsCurrency = (req.query.vs || 'USD').toUpperCase();
        const snapshot = getLatestSnapshot(vsCurrency);
        if (!snapshot) {
            // If no snapshot exists, return empty data
            return res.json({
                vsCurrency,
                timestamp: Date.now(),
                items: []
            });
        }
        res.json(snapshot);
    }
    catch (error) {
        console.error('Error fetching market snapshot:', error);
        res.status(500).json({ error: 'Failed to fetch market data' });
    }
});
export default router;
