import { Router } from 'express';
import pool from '../config/db.js';
import { authGuard } from '../middleware/auth.js';
const router = Router();
async function getCurrentUserId(req) {
    const user = req.user;
    if (!user)
        return null;
    const [u] = await pool.query(`SELECT id FROM users WHERE uid = :uid LIMIT 1`, { uid: user.sub });
    if (!u || !u.length)
        return null;
    return Number(u[0].id);
}
function onlyDigits(val) {
    return String(val ?? '').replace(/\D/g, '');
}
router.get('/methods', authGuard, async (req, res) => {
    const userId = await getCurrentUserId(req);
    if (!userId)
        return res.status(401).json({ error: 'Unauthorized' });
    const [cards] = await pool.query(`SELECT id, name_on_card as nameOnCard, brand, last4, exp_month as expMonth, exp_year as expYear FROM payment_cards WHERE user_id = :uid ORDER BY created_at DESC`, { uid: userId });
    const [banks] = await pool.query(`SELECT id, account_holder as accountHolder, bank_name as bankName, iban, swift, routing, last4 FROM payment_banks WHERE user_id = :uid ORDER BY created_at DESC`, { uid: userId });
    res.json({ cards, banks });
});
router.post('/cards', authGuard, async (req, res) => {
    const userId = await getCurrentUserId(req);
    if (!userId)
        return res.status(401).json({ error: 'Unauthorized' });
    const nameOnCard = String(req.body?.nameOnCard || '').trim();
    const numberDigits = onlyDigits(req.body?.number);
    const mmDigits = onlyDigits(req.body?.expMonth).slice(0, 2);
    const yyDigits = onlyDigits(req.body?.expYear).slice(0, 4);
    if (!nameOnCard || numberDigits.length < 12)
        return res.status(400).json({ error: 'Invalid card details' });
    const mmNum = Number(mmDigits);
    if (!mmDigits || mmNum < 1 || mmNum > 12)
        return res.status(400).json({ error: 'Invalid expiration month' });
    if (!yyDigits || yyDigits.length !== 4)
        return res.status(400).json({ error: 'Invalid expiration year' });
    const last4 = numberDigits.slice(-4);
    await pool.query(`INSERT INTO payment_cards (user_id, name_on_card, brand, last4, exp_month, exp_year) VALUES (:uid, :name, NULL, :last4, :mm, :yy)`, { uid: userId, name: nameOnCard, last4, mm: mmDigits.padStart(2, '0'), yy: yyDigits.padStart(4, '0') });
    res.status(201).json({ ok: true });
});
router.delete('/cards/:id', authGuard, async (req, res) => {
    const userId = await getCurrentUserId(req);
    if (!userId)
        return res.status(401).json({ error: 'Unauthorized' });
    const id = Number(req.params.id);
    await pool.query(`DELETE FROM payment_cards WHERE id = :id AND user_id = :uid`, { id, uid: userId });
    res.json({ ok: true });
});
router.post('/banks', authGuard, async (req, res) => {
    const userId = await getCurrentUserId(req);
    if (!userId)
        return res.status(401).json({ error: 'Unauthorized' });
    const accountHolder = String(req.body?.accountHolder || '').trim();
    const bankName = String(req.body?.bankName || '').trim();
    const iban = String(req.body?.iban || '').trim() || null;
    const swift = String(req.body?.swift || '').trim() || null;
    const routing = onlyDigits(req.body?.routing) || null;
    if (!accountHolder || !bankName)
        return res.status(400).json({ error: 'Missing fields' });
    const last4 = onlyDigits(iban || routing || '').slice(-4) || null;
    await pool.query(`INSERT INTO payment_banks (user_id, account_holder, bank_name, iban, swift, routing, last4) VALUES (:uid, :holder, :bank, :iban, :swift, :routing, :last4)`, { uid: userId, holder: accountHolder, bank: bankName, iban, swift, routing, last4 });
    res.status(201).json({ ok: true });
});
router.delete('/banks/:id', authGuard, async (req, res) => {
    const userId = await getCurrentUserId(req);
    if (!userId)
        return res.status(401).json({ error: 'Unauthorized' });
    const id = Number(req.params.id);
    await pool.query(`DELETE FROM payment_banks WHERE id = :id AND user_id = :uid`, { id, uid: userId });
    res.json({ ok: true });
});
export default router;
