import { Router } from 'express';
import multer from 'multer';
import path from 'path';
import fs from 'fs';
import { fileURLToPath } from 'url';
import { authGuard } from '../middleware/auth.js';
const router = Router();
const __dirname = path.dirname(fileURLToPath(import.meta.url));
const kycBaseDir = path.join(__dirname, '..', '..', 'assets', 'kyc');
function ensureDir(dir) {
    try {
        fs.mkdirSync(dir, { recursive: true });
    }
    catch { }
}
function safeName(name) {
    return path.basename(name).replace(/[^a-zA-Z0-9._-]/g, '_');
}
const storage = multer.diskStorage({
    destination: (req, file, cb) => {
        const user = req.user;
        const category = ['idDocument', 'selfie', 'proofOfAddress'].includes(file.fieldname) ? file.fieldname : 'other';
        const userDir = path.join(kycBaseDir, user.sub, category);
        ensureDir(userDir);
        cb(null, userDir);
    },
    filename: (_req, file, cb) => {
        const timestamp = Date.now();
        cb(null, `${timestamp}-${safeName(file.originalname)}`);
    }
});
const upload = multer({
    storage,
    limits: { fileSize: 10 * 1024 * 1024 },
    fileFilter: (_req, file, cb) => {
        const allowed = ['image/jpeg', 'image/png', 'image/webp', 'application/pdf'];
        if (allowed.includes(file.mimetype))
            return cb(null, true);
        cb(new Error('Unsupported file type'));
    }
});
// POST /api/kyc/upload - current user uploads categorized KYC docs (one per category)
router.post('/upload', authGuard, upload.fields([
    { name: 'idDocument', maxCount: 1 },
    { name: 'selfie', maxCount: 1 },
    { name: 'proofOfAddress', maxCount: 1 },
]), async (req, res) => {
    const user = req.user;
    const uid = user.sub;
    const files = req.files;
    const toItems = (category) => {
        const arr = files?.[category] || [];
        return arr.map(f => ({ name: f.filename, url: `/assets/kyc/${uid}/${category}/${f.filename}`, size: f.size, uploadedAt: new Date().toISOString() }));
    };
    res.json({
        categories: {
            idDocument: toItems('idDocument'),
            selfie: toItems('selfie'),
            proofOfAddress: toItems('proofOfAddress'),
        }
    });
});
export default router;
